<?php
declare(strict_types=1);

final class CapabilitiesController {
  /**
   * Lists the capability catalog visible to the caller.
   * Returns global capabilities (org_id NULL) plus org-specific capabilities
   * for the requested org (default: caller org).
   */
  public static function list(): void {
    $u = require_login();
    $pdo = db();
    $orgId = (string)($_GET['org_id'] ?? $u['org_id']);
    if ($u['role'] !== 'admin' && $orgId !== $u['org_id']) {
      api_error(403, 'Cannot list other org capabilities');
    }
    $stmt = $pdo->prepare("SELECT id, org_id, key, label, description, color, sort_order, is_active
      FROM capability
      WHERE is_active=true AND (org_id IS NULL OR org_id=:oid)
      ORDER BY org_id NULLS FIRST, sort_order ASC, label ASC");
    $stmt->execute([':oid'=>$orgId]);
    $rows = $stmt->fetchAll();
    api_json(['ok'=>true, 'capabilities'=>$rows]);
  }

  public static function create(): void {
    $u = require_role(['admin','el']);
    $in = json_input();
    $scope = (string)($in['scope'] ?? 'org'); // 'global' or 'org'
    $orgId = (string)($in['org_id'] ?? $u['org_id']);
    if ($scope === 'global') {
      if ($u['role'] !== 'admin') api_error(403, 'Only admin can create global capabilities');
      $orgId = '';
    } else {
      if ($u['role'] !== 'admin' && $orgId !== $u['org_id']) api_error(403, 'Cannot create for other org');
    }

    $key = strtolower(trim((string)($in['key'] ?? '')));
    $label = trim((string)($in['label'] ?? ''));
    if ($key === '' || $label === '') api_error(400, 'key and label required');
    if (!preg_match('/^[a-z0-9_\-\.]{2,40}$/', $key)) api_error(400, 'invalid key');

    $desc = trim((string)($in['description'] ?? ''));
    $color = trim((string)($in['color'] ?? ''));
    $sort = (int)($in['sort_order'] ?? 100);

    $pdo = db();
    $stmt = $pdo->prepare("INSERT INTO capability(org_id, key, label, description, color, sort_order, is_active)
      VALUES(" . ($scope==='global' ? "NULL" : ":oid") . ", :k, :l, :d, :c, :s, true) RETURNING id");
    $params = [':k'=>$key, ':l'=>$label, ':d'=>$desc ?: null, ':c'=>$color ?: null, ':s'=>$sort];
    if ($scope !== 'global') $params[':oid'] = $orgId;
    try {
      $stmt->execute($params);
    } catch (Throwable $e) {
      api_error(400, 'Capability key already exists');
    }
    $id = (int)$stmt->fetchColumn();

    EventsController::emitGlobal('capabilities.updated', ['org_id'=>($scope==='global'?null:$orgId)]);
    api_json(['ok'=>true, 'id'=>$id], 201);
  }

  public static function update(int $id): void {
    $u = require_role(['admin','el']);
    $in = json_input();
    $pdo = db();
    $row = $pdo->prepare('SELECT id, org_id FROM capability WHERE id=:id');
    $row->execute([':id'=>$id]);
    $cap = $row->fetch();
    if (!$cap) api_error(404, 'Not found');
    $capOrg = $cap['org_id'];
    if ($capOrg === null) {
      if ($u['role'] !== 'admin') api_error(403, 'Only admin can update global capabilities');
    } else {
      if ($u['role'] !== 'admin' && $capOrg !== $u['org_id']) api_error(403, 'Not allowed');
    }

    $label = trim((string)($in['label'] ?? ''));
    $desc = array_key_exists('description', $in) ? trim((string)$in['description']) : null;
    $color = array_key_exists('color', $in) ? trim((string)$in['color']) : null;
    $sort = array_key_exists('sort_order', $in) ? (int)$in['sort_order'] : null;
    $active = array_key_exists('is_active', $in) ? (bool)$in['is_active'] : null;

    $sets = [];
    $params = [':id'=>$id];
    if ($label !== '') { $sets[]='label=:l'; $params[':l']=$label; }
    if ($desc !== null) { $sets[]='description=:d'; $params[':d']=$desc===''?null:$desc; }
    if ($color !== null) { $sets[]='color=:c'; $params[':c']=$color===''?null:$color; }
    if ($sort !== null) { $sets[]='sort_order=:s'; $params[':s']=$sort; }
    if ($active !== null) { $sets[]='is_active=:a'; $params[':a']=$active; }
    $sets[]='updated_at=now()';

    $pdo->prepare('UPDATE capability SET ' . implode(',', $sets) . ' WHERE id=:id')->execute($params);
    EventsController::emitGlobal('capabilities.updated', ['org_id'=>$capOrg]);
    api_json(['ok'=>true]);
  }

  public static function remove(int $id): void {
    $u = require_role(['admin','el']);
    $pdo = db();
    $row = $pdo->prepare('SELECT id, org_id FROM capability WHERE id=:id');
    $row->execute([':id'=>$id]);
    $cap = $row->fetch();
    if (!$cap) api_error(404, 'Not found');
    $capOrg = $cap['org_id'];
    if ($capOrg === null) {
      if ($u['role'] !== 'admin') api_error(403, 'Only admin can remove global capabilities');
    } else {
      if ($u['role'] !== 'admin' && $capOrg !== $u['org_id']) api_error(403, 'Not allowed');
    }
    $pdo->prepare('UPDATE capability SET is_active=false, updated_at=now() WHERE id=:id')->execute([':id'=>$id]);
    EventsController::emitGlobal('capabilities.updated', ['org_id'=>$capOrg]);
    api_json(['ok'=>true]);
  }
}
